from datetime import datetime

from flask import render_template, request, url_for, flash, current_app
from werkzeug.utils import redirect
from flask_login import current_user, login_user, logout_user
from flask_babel import _

from app.forms import MessageForm
from app.models import Message
from app import db
from app.models import User
from app.models import CocktailDTO
from app.forms import SimpleForm, EvalForm, LoginForm, RegistrationForm, ResetPasswordForm
from app import app, cocktailRecommender
from app.forms import ResetPasswordRequestForm
from app.email import send_password_reset_email

from flask_paginate import Pagination, get_page_args
from flask_login import login_required
from flask import request
from werkzeug.urls import url_parse


@app.route('/')
@app.route('/index')
@login_required
def index():
    return render_template('index.html', title='Home')


@app.route('/cocktail/<name>?<ask_eval>', methods=["GET", "POST"])
def cocktail(name, ask_eval):
    for c in cocktailRecommender.case_base.get_all_cocktails():
        print(c.ingredient_quantity_unit)
        print('----------')


    form = EvalForm()
    cocktail = cocktailRecommender.get_recommended_cocktail() if ask_eval == 'True' else cocktailRecommender.get_cocktail(name)
    if cocktail is not None:
        print(cocktail.ingredient_quantity_unit)
        cocktailDTO = CocktailDTO(cocktail)
        if form.validate_on_submit():
            value = request.form.getlist('options')
            if len(value) == 0:
                error = 'You must select an opinion.'
                return render_template('cocktail.html', cocktail=cocktailDTO, ask_eval=str(ask_eval), form=form, error=error)
            cocktailRecommender.set_user_evaluation(value[0])  # Process user evaluation
            flash('Thanks for your opinion! we are going to keep improving :)')
            return render_template('cocktail.html', cocktail=cocktailDTO, ask_eval=str(False), form=form)
        else:
            return render_template('cocktail.html', cocktail=cocktailDTO, ask_eval=str(ask_eval), form=form)
    else:
        return render_template('404.html', name=name)


@app.errorhandler(404)
def not_found_error(error):
    return render_template('404.html', name=''), 404


@app.errorhandler(500)
def internal_error(error):
    db.session.rollback()
    return render_template('500.html', name=''), 500


def get_cocktail_pagination(cocktail_list, per_page=10, offset=0):
    return cocktail_list[offset: offset + per_page]


@app.route('/cocktails')
def cocktails():
    # Get cocktails from case base
    cocktail_list = cocktailRecommender.get_all_cocktails()

    # Preprocess them for jinja2 template
    cocktail_list = [CocktailDTO(c) for c in cocktail_list]
    cocktail_list = sorted(cocktail_list, key=lambda x: x.name)

    # Paginate cocktail list
    page, per_page, offset = get_page_args(page_parameter='page',
                                           per_page_parameter='per_page')
    total = len(cocktail_list)
    pagination_cocktails = get_cocktail_pagination(cocktail_list, offset=offset, per_page=per_page)
    pagination = Pagination(page=page, per_page=per_page, total=total,
                            css_framework='bootstrap4')

    return render_template('cocktails.html',
                           cocktails=pagination_cocktails,
                           page=page,
                           per_page=per_page,
                           pagination=pagination,
                           )


@app.route('/specially4u', methods=["GET", "POST"])
def specially4u():
    form = SimpleForm()
    error = ""
    if form.validate_on_submit():
        if len(form.fruits_cb.data) + len(form.alco_cb.data) + len(form.nonalco_cb.data) + len(form.others_cb.data) < 2:
            error = "You must selected at least two ingredients"
            return render_template('specially4u.html', form=form, error=error)
        else:
            # Get user input ingredients
            response = [form.fruits_cb.data, form.alco_cb.data, form.nonalco_cb.data, form.others_cb.data]
            user_query = [val.lower() for sublist in response for val in sublist]
            print('USER QUERY ', user_query)

            # Search recommended cocktail
            cocktail = cocktailRecommender.get_recommendation(user_query)

            # Check if a recommendation exists
            if cocktail.name is None:
                return render_template('no_recommendation.html', user_query=user_query)
            else:
                return redirect(url_for('cocktail', name=cocktail.name, ask_eval=True))
    return render_template('specially4u.html', form=form, error=error)


@app.route('/login', methods=['GET', 'POST'])
def login():
    if current_user.is_authenticated:
        return redirect(url_for('index'))
    form = LoginForm()
    if form.validate_on_submit():
        user = User.query.filter_by(username=form.username.data).first()
        if user is None or not user.check_password(form.password.data):
            flash('Invalid username or password')
            return redirect(url_for('login'))
        login_user(user, remember=form.remember_me.data)
        next_page = request.args.get('next')
        if not next_page or url_parse(next_page).netloc != '':
            next_page = url_for('index')
        return redirect(next_page)
    return render_template('login.html', title='Sign In', form=form)


@app.route('/register', methods=['GET', 'POST'])
def register():
    if current_user.is_authenticated:
        return redirect(url_for('index'))
    form = RegistrationForm()
    if form.validate_on_submit():
        user = User(username=form.username.data, email=form.email.data)
        user.set_password(form.password.data)
        db.session.add(user)
        db.session.commit()
        flash(_('Congratulations, you are now a registered user!'))
        return redirect(url_for('login'))
    return render_template('register.html', title='Register', form=form)


@app.route('/logout')
def logout():
    logout_user()
    return redirect(url_for('index'))


@app.route('/help')
def help():
    return render_template('help.html', title='help')


@app.route('/reset_password_request', methods=['GET', 'POST'])
def reset_password_request():
    if current_user.is_authenticated:
        return redirect(url_for('index'))
    form = ResetPasswordRequestForm()
    if form.validate_on_submit():
        user = User.query.filter_by(email=form.email.data).first()
        if user:
            send_password_reset_email(user)
        flash('Check your email for the instructions to reset your password')
        return redirect(url_for('login'))
    return render_template('reset_password_request.html',
                           title='Reset Password', form=form)


@app.route('/reset_password/<token>', methods=['GET', 'POST'])
def reset_password(token):
    if current_user.is_authenticated:
        return redirect(url_for('index'))
    user = User.verify_reset_password_token(token)
    if not user:
        return redirect(url_for('index'))
    form = ResetPasswordForm()
    if form.validate_on_submit():
        user.set_password(form.password.data)
        db.session.commit()
        flash('Your password has been reset.')
        return redirect(url_for('login'))
    return render_template('reset_password.html', form=form)


@app.route('/message_us', methods=['GET', 'POST'])
@login_required
def message_us():
    form = MessageForm()
    user = User.query.filter_by(username="barkeeper").first_or_404()
    print(user.username)
    if form.validate_on_submit():
        print(form.message.data)
        msg = Message(author=current_user, recipient=user,
                      body=form.message.data)
        db.session.add(msg)
        db.session.commit()
        flash(_('Your message has been sent.'))
        return redirect(url_for('index'))
    return render_template('send_message.html', title=_('Send Message'),
                           form=form)


@app.route('/messages')
@login_required
def messages():
    current_user.last_message_read_time = datetime.utcnow()
    db.session.commit()
    page = request.args.get('page', 1, type=int)
    messages = current_user.messages_received.order_by(Message.timestamp.desc())\
        .paginate(page, current_app.config['MESSAGES_PER_PAGE'], False)
    next_url = url_for('messages', page=messages.next_num) if messages.has_next else None
    prev_url = url_for('messages', page=messages.prev_num) if messages.has_prev else None
    return render_template('messages.html', messages=messages.items,
                           next_url=next_url, prev_url=prev_url)


if __name__ == '__main__':
    app.run(debug=True)
